import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Check, Shield, Leaf, Calendar, Download, RotateCcw } from "lucide-react";
import { WizardData } from "@/lib/types";
import { calculateFinalQuote, getServiceDisplayName, getTimelineDisplayName } from "@/lib/quote-calculator";

interface QuoteDisplayProps {
  data: WizardData;
  onBack: () => void;
}

export function QuoteDisplay({ data, onBack }: QuoteDisplayProps) {
  const quote = calculateFinalQuote(data);

  const downloadQuote = () => {
    // TODO: Implement PDF generation
    alert("Quote download functionality will be implemented.");
  };

  const startOver = () => {
    if (confirm("Are you sure you want to start a new quote? All current information will be lost.")) {
      window.location.reload();
    }
  };

  return (
    <div className="p-8">
      <div className="text-center mb-8">
        <div className="w-20 h-20 bg-eco-primary rounded-full flex items-center justify-center mx-auto mb-4">
          <Check className="text-white text-3xl w-8 h-8" />
        </div>
        <h3 className="text-2xl font-bold text-eco-secondary mb-2">Your instant quote is ready!</h3>
        <p className="text-gray-600">We've calculated the best service option for your needs</p>
      </div>

      <div className="max-w-3xl mx-auto">
        {/* Quote Summary Card */}
        <div className="bg-gradient-to-br from-eco-primary to-green-600 rounded-2xl p-8 text-white mb-8">
          <div className="grid grid-cols-1 md:grid-cols-2 gap-8">
            <div>
              <h4 className="text-lg font-semibold mb-4">Service Summary</h4>
              <div className="space-y-3">
                <div className="flex justify-between">
                  <span>Service Type:</span>
                  <span className="font-semibold">{getServiceDisplayName(data.serviceType || "")}</span>
                </div>
                <div className="flex justify-between">
                  <span>Volume:</span>
                  <span className="font-semibold">{quote.totalBoxes} boxes ({quote.totalWeight} lbs)</span>
                </div>
                <div className="flex justify-between">
                  <span>Timeline:</span>
                  <span className="font-semibold">{getTimelineDisplayName(data.timeline || "")}</span>
                </div>
                <div className="flex justify-between">
                  <span>Location:</span>
                  <span className="font-semibold">{data.zipCode}</span>
                </div>
              </div>
            </div>
            <div className="text-center md:text-right">
              <div className="text-sm opacity-90 mb-2">Estimated Total</div>
              <div className="text-4xl font-bold mb-2">${quote.price}</div>
              <div className="text-sm opacity-90">Includes certificate & witnessed destruction</div>
              <div className="mt-4 bg-white bg-opacity-20 rounded-lg p-3">
                <div className="text-sm">You save up to</div>
                <div className="text-2xl font-bold">${quote.savings}</div>
                <div className="text-xs opacity-90">vs. other providers</div>
              </div>
            </div>
          </div>
        </div>

        {/* What's Included */}
        <div className="grid grid-cols-1 md:grid-cols-2 gap-6 mb-8">
          <Card className="border-green-200">
            <CardHeader>
              <CardTitle className="flex items-center text-eco-secondary">
                <Shield className="w-5 h-5 text-eco-primary mr-2" />
                Security Features
              </CardTitle>
            </CardHeader>
            <CardContent>
              <ul className="space-y-2 text-sm text-gray-600">
                <li className="flex items-center">
                  <Check className="text-eco-primary mr-2 w-4 h-4" />
                  Witnessed destruction
                </li>
                <li className="flex items-center">
                  <Check className="text-eco-primary mr-2 w-4 h-4" />
                  Certificate of destruction
                </li>
                <li className="flex items-center">
                  <Check className="text-eco-primary mr-2 w-4 h-4" />
                  Chain of custody tracking
                </li>
                <li className="flex items-center">
                  <Check className="text-eco-primary mr-2 w-4 h-4" />
                  HIPAA compliance
                </li>
              </ul>
            </CardContent>
          </Card>
          
          <Card className="border-green-200">
            <CardHeader>
              <CardTitle className="flex items-center text-eco-secondary">
                <Leaf className="w-5 h-5 text-eco-primary mr-2" />
                Environmental Impact
              </CardTitle>
            </CardHeader>
            <CardContent>
              <ul className="space-y-2 text-sm text-gray-600">
                <li className="flex items-center">
                  <Check className="text-eco-primary mr-2 w-4 h-4" />
                  100% recycled into new paper
                </li>
                <li className="flex items-center">
                  <Check className="text-eco-primary mr-2 w-4 h-4" />
                  Carbon-neutral transportation
                </li>
                <li className="flex items-center">
                  <Check className="text-eco-primary mr-2 w-4 h-4" />
                  Zero landfill waste
                </li>
                <li className="flex items-center">
                  <Check className="text-eco-primary mr-2 w-4 h-4" />
                  Certified green processing
                </li>
              </ul>
            </CardContent>
          </Card>
        </div>

        {/* Success Message */}
        <div className="bg-green-50 border border-green-200 rounded-xl p-6 mb-8">
          <div className="text-center">
            <div className="w-16 h-16 bg-green-500 rounded-full flex items-center justify-center mx-auto mb-4">
              <Check className="text-white w-8 h-8" />
            </div>
            <h4 className="font-semibold text-green-800 mb-2 text-lg">Lead Successfully Submitted!</h4>
            <p className="text-green-700 mb-4">Your information has been sent to our team and you should receive a confirmation email shortly.</p>
          </div>
        </div>

        {/* Next Steps */}
        <div className="bg-gray-50 rounded-xl p-6 mb-8">
          <h4 className="font-semibold text-eco-secondary mb-4">What happens next?</h4>
          <div className="grid grid-cols-1 md:grid-cols-3 gap-4 text-sm">
            <div className="text-center">
              <div className="w-12 h-12 bg-eco-primary text-white rounded-full flex items-center justify-center mx-auto mb-2">
                1
              </div>
              <div className="font-semibold">Email Sent</div>
              <div className="text-gray-600">Check your email for confirmation</div>
            </div>
            <div className="text-center">
              <div className="w-12 h-12 bg-eco-primary text-white rounded-full flex items-center justify-center mx-auto mb-2">
                2
              </div>
              <div className="font-semibold">Provider Contact</div>
              <div className="text-gray-600">We'll call within 24 hours</div>
            </div>
            <div className="text-center">
              <div className="w-12 h-12 bg-eco-primary text-white rounded-full flex items-center justify-center mx-auto mb-2">
                3
              </div>
              <div className="font-semibold">Service Delivery</div>
              <div className="text-gray-600">Schedule and complete service</div>
            </div>
          </div>
        </div>

        {/* Action Buttons */}
        <div className="flex flex-col md:flex-row gap-4 justify-center">
          <Button 
            onClick={downloadQuote}
            className="bg-eco-primary text-white px-8 py-4 hover:bg-green-600"
          >
            <Download className="w-4 h-4 mr-2" />
            Download Quote
          </Button>
          <Button 
            onClick={startOver}
            variant="outline"
            className="border-eco-primary text-eco-primary px-8 py-4 hover:bg-eco-light"
          >
            <RotateCcw className="w-4 h-4 mr-2" />
            Start New Quote
          </Button>
        </div>
      </div>
    </div>
  );
}
